######################################################################################
# This code (Version 1.5) is provided for educational purposes.
# If you use, adapt, or redistribute this code, please cite the source as follows:
# Dun-Yen Kang at National Taiwan University, http://www.mmlab-ntu.tw , September 2024 
######################################################################################

import numpy as np
from scipy.integrate import solve_ivp
import matplotlib.pyplot as plt

# Constants
A = 0.01  # Cross section area of cylinder, unit=[m^2]
n = 100  # Mole of gas
R = 8.314  # Gas constant, unit=[J/K/mol]
Cv = 1.5 * R  # Constant-volume heat capacity, unit=[J/mol/K]
Tini = 298  # Initial temperature, unit=[K]
Vini = 1 # initial volume in [m^3]
Pext = 1 * 10**5  # External pressure, unit=[Pa]
g = 9.8  # Gravity, unit=[m/s^2]

m = float(input('Specify mass of piston [kg] (5 is suggested value): '))
# Mass of piston, unit=[kg]
f = float(input('Specify friction coefficient btween piston and wall in [N*s/m] (2 is suggested value): '))
# Coefficient of friction, unit=[N*s/m]
t_final = float(input('Speficy the duration of the process in [s] (50 is suggested value): '))
# DPI of the figures
fig_dpi = int(input('Specify DPI of the output figures (150 is suggested value): '))

# for massless case, m=1e-3 [kg], f=3 [N*s/m], t_final =2 

t_step = 5000

# Differential equations for adiabatic expansion
def adiabatic_exp(t, y):
    u, V, T = y  # Unpacking velocity, volume, and temperature
    dudt = A * n * R * T / (m * V) - Pext * A / m - g - f * u / m
    dVdt = A * u
    dTdt = (-Pext * A * u - m * u * dudt - m * g * u) / (n * Cv)
    return [dudt, dVdt, dTdt]

# Initial conditions and time span
y0 = [0, Vini, Tini]
tspan = np.linspace(0, t_final, t_step)

# Solve ODE
sol = solve_ivp(adiabatic_exp, [tspan[0], tspan[-1]], y0, t_eval=tspan, method='RK45', rtol=1e-5)

# Extract results
t = sol.t # time
u = sol.y[0] # velocity of piston
V = sol.y[1] # volume of cylinder
T = sol.y[2] # temperature of gas inside cylinder
P = n * R * T / V  # pressure of gas inside cylinder
h = V / A # position of piston
KE = 0.5 * m * u**2 # kinetic energy of piston (or the entire system)
PE = m * g * h # potential energy of piston (or the entire system)

# Additional calculations for work rate, entropy change rate, etc.
V_plus = V[1:]
V_minus = V[:-1]
u_plus = u[1:]
u_minus = u[:-1]
h_plus = h[1:]
h_minus = h[:-1]
P_plus = P[1:]
P_minus = P[:-1]
T_plus = T[1:]
T_minus = T[:-1]
t_plus = t[1:]
t_minus = t[:-1]

P_avg = 0.5 * (P_plus + P_minus)
T_avg = 0.5 * (T_plus + T_minus)
V_avg = 0.5 * (V_plus + V_minus)
u_avg = 0.5 * (u_plus + u_minus)
t_avg = 0.5 * (t_plus + t_minus)

time_step_size = t_final / t_step

work_outside_rate = -Pext * (V_plus - V_minus) / time_step_size # work rate for the system including the pistion
work_inside_rate = -P_avg * (V_plus - V_minus) / time_step_size # work rate for the system of gas only
U_rate = n * Cv * (T_plus - T_minus) / time_step_size
heat_inside_rate = U_rate - work_inside_rate
heat_friction_rate = f * u_avg **2 # heat rate from the negative work of friction fource 
S_rate = (n * Cv * (T_plus - T_minus) / T_avg + n * R * (V_plus - V_minus) / V_avg) / time_step_size
S_gen_rate = S_rate
Qf_over_T_rate = f * u_avg **2 / T_avg

work_outside_total = np.sum(work_outside_rate) * time_step_size
work_inside_total = np.sum(work_inside_rate) * time_step_size
heat_inside_total = np.sum(heat_inside_rate) * time_step_size
heat_friction_total =  np.sum(heat_friction_rate) * time_step_size
PE_total = m * g * (h[-1] - h[0])
KE_total = 0.5 * m * (u[-1]**2 - u[0]**2)
IE_total = n * Cv * (T[-1] - T[0])
S_total = np.sum(S_rate) * 0.01
S_gen_total = np.sum(S_gen_rate) * time_step_size
Qf_over_T_total = np.sum(Qf_over_T_rate) * time_step_size

print('Final V =', V[t_step - 1],'[m^3]')
print('Final T =', T[t_step - 1],'[m^3]')
print('Final P =', n * R * T[t_step - 1] / V[t_step - 1],'[Pa]')
print('W done by the system including pistion =', work_outside_total,'[J]')
print('W done by the system with gas only (excluding pistion) =', work_inside_total,'[J]')
print('Q for the system excluding pistion =', heat_inside_total,'[J]')
print('Qf (heat from friction) =', heat_friction_total,'[J]')
print('Delta U =', IE_total,'[J]')
print('Delta Ep =', PE_total,'[J]')
print('Delta Ek =', KE_total,'[J]')
print('Delta S =', S_total,'[J/K]')
print('Entropy generation from the system including piston =', S_gen_total,'[J/K]')
print('Integral of rate of Qf/T over t =', Qf_over_T_total,'[J/K]')

# Plotting
plt.figure(dpi = fig_dpi)
plt.plot(t, u)
plt.xlabel('t (s)')
plt.ylabel('velocity (m/s)')
plt.grid()

plt.figure(dpi = fig_dpi)
plt.plot(t, V)
plt.xlabel('t (s)')
plt.ylabel('V (m$^3$)')
plt.grid()

plt.figure(dpi = fig_dpi)
plt.plot(t, T)
plt.xlabel('t (s)')
plt.ylabel('T (K)')
plt.grid()

plt.figure(dpi = fig_dpi)
plt.plot(t_avg, heat_friction_rate)
plt.xlabel('t (s)')
plt.ylabel('Rate of $Q_f$ (J/s)')
plt.grid()

plt.figure(dpi = fig_dpi)
plt.plot(t_avg, S_gen_rate)
plt.xlabel('t (s)')
plt.ylabel('Entropy generation rate (J/K/s)')
plt.grid()

plt.figure(dpi = fig_dpi)
plt.plot(t_avg, Qf_over_T_rate)
plt.xlabel('t (s)')
plt.ylabel('Rate of $Q_f$ / T (J/K/s)')
plt.grid()

plt.show()