######################################################################################
# This code (Version 1.6) is provided for educational purposes.
# If you use, adapt, or redistribute this code, please cite the source as follows:
# Dun-Yen Kang at National Taiwan University, http://www.mmlab-ntu.tw , September 2024 
######################################################################################

import numpy as np
import matplotlib.pyplot as plt

R = 8.314

def kappa(omega):
    return 0.37464 + 1.5422 * omega - 0.26992 * omega**2

def alpha(t, tc, omega):
    return (1 + kappa(omega) * (1 - np.sqrt(t / tc)))**2

def b(tc, pc):
    return 0.0778 * R * tc / pc

def a(t, tc, pc, omega):
    return 0.45724 * R**2 * tc**2 / pc * alpha(t, tc, omega)

def P_PR_EOS(t, tc, pc, omega, v):
    return R * t / (v - b(tc, pc)) - a(t, tc, pc, omega) / (v * (v + b(tc, pc)) + b(tc, pc) * (v - b(tc, pc)))

def pmax_min(p, v): #find the local max and min for the PV curve 
    pks_max = np.r_[True, p[1:] > p[:-1]] & np.r_[p[:-1] > p[1:], True]
    pks_min = np.r_[True, p[1:] < p[:-1]] & np.r_[p[:-1] < p[1:], True]
    pmax = np.sum(p * pks_max)
    vmax = np.sum(v * pks_max)
    pmin = np.sum(p * pks_min)
    vmin = np.sum(v * pks_min)
    return pmax, vmax, pmin, vmin

def finding_vl_vv(p, t, tc, pc, omega): #solve the PREOS in a form of cubic equation at given T, P
    A = a(t, tc, pc, omega) * p / R**2 / t**2
    B = b(tc, pc) * p / R / t
    al = -(1 - B)
    be = (A - 2 * B - 3 * B**2)
    ga = -(A * B - B**2 - B**3)
    roots = np.roots([1, al, be, ga])
    return roots * R * t / p

def Z(p, v, t):
    return p * v / R / t

def lnphi(v, p, t, tc, pc, omega): #compute natural log phi from PREOS
    Z_value = Z(p, v, t)
    b_value = b(tc, pc)

    if Z_value <= 0 or np.isclose(Z_value, b_value * p / R / t):
        # Handle the case when the argument of the logarithm becomes invalid
        return np.nan  # or any other appropriate value in this case

    term1 = (Z_value - 1)
    term2 = np.log(Z_value - b_value * p / R / t)
    term3 = a(t, tc, pc, omega) / b(tc, pc) / R / t / 2.828 * np.log((Z_value + 2.414 * b_value * p / R / t) / (Z_value - 0.414 * b_value * p / R / t))

    return term1 - term2 - term3

def H_depart(v, p, t, tc, pc, omega): # to compute enthalpy departure function
    dadt = (a(t, tc, pc, omega) - a(t - 0.1, tc, pc, omega)) / 0.1
    cap_B = b(tc, pc) * p / R / t
    term1 = R * t * (Z(p, v, t) - 1)
    term2 = (t * dadt - a(t, tc, pc, omega)) / (2 * np.sqrt(2) * b(tc, pc))
    term3 = np.log((Z(p, v, t) + (1 + np.sqrt(2)) * cap_B) / (Z(p, v, t) + (1 - np.sqrt(2)) * cap_B))
    return term1 + term2 * term3

def S_depart(v, p, t, tc, pc, omega): # to compute entropy departure function
    dadt = (a(t, tc, pc, omega) - a(t - 0.1, tc, pc, omega)) / 0.1
    cap_B = b(tc, pc) * p / R / t
    term1 = R * np.log(Z(p, v, t) - cap_B)
    term2 = dadt / (2 * np.sqrt(2) * b(tc, pc))
    term3 = np.log((Z(p, v, t) + (1 + np.sqrt(2)) * cap_B) / (Z(p, v, t) + (1 - np.sqrt(2)) * cap_B))
    return term1 + term2 * term3

Tempc = float(input('Enter critical temperature in [K] of the target fluid (154.6 for Oxygen): '))
Pressc = float(input('Enter critical pressure in [Pa] of the target fluid (5.046e6 for Oxygen): '))
Omega = float(input('Enter acentric factor, ω, of the target fluid (0.021 for Oxygen): '))
Temp = float(input('Specify target T in [K]: '))
Press = float(input('Specify target P  in [Pa]: '))
# DPI of the figures
fig_dpi = int(input('Specify DPI of the output figures (150 is suggested value): '))

vol = finding_vl_vv(Press, Temp, Tempc, Pressc, Omega)
vol = [v for v in vol if isinstance(v, (int, float)) or (isinstance(v, complex) and v.imag == 0)]
#remove the volume values that complex number
vol = [v.real for v in vol]
#make the volume values a real number (remove the imanginary part, even if it is zero)

Vstart = b(Tempc, Pressc) * 1.05  # for plotting
V = np.logspace(np.log10(Vstart), -1, 500)  # for plotting
P = np.zeros(500)  # for plotting
for i in range(500):
    P[i] = P_PR_EOS(Temp, Tempc, Pressc, Omega, V[i])

P_log = np.zeros(500)  # plot p in log scale (plot in right bottom)

for i in range(500):
    if P[i] < 1E1:
        P_log[i] = 1E1  # avoid problem of plotting P with a negative value
    else:
        P_log[i] = P[i]

if Temp >= Tempc:
    
    print('V_L = NaN')
    print(f'V_V = {vol[0]} m^3/mol')
    
    print('Z_L = NaN')
    print('Z_V =', Z(Press, vol[0], Temp))

    print('H_departure_L = NaN')
    print(f'H_departure_V = {H_depart(vol[0], Press, Temp, Tempc, Pressc, Omega)} J/mol')

    print('S_departure_L = NaN')
    print(f'S_departure_V = {S_depart(vol[0], Press, Temp, Tempc, Pressc, Omega)} J/mol/K')

    print('φ_L = NaN')
    print(f'φ_V = {np.exp(lnphi(vol[0], Press, Temp, Tempc, Pressc, Omega))}')

    print('The stable phase is a supercritical fluid. The vapor-to-liquid transition does not exist at the given temperature.')
    print('Warning: Part of the curve in the logarithmic scale figure may be truncated to prevent errors from displaying negative values.')

    plt.figure(dpi = fig_dpi)
    plt.semilogx(V, P, [vol[0]], [Press], 'o', markersize=10)
    plt.xlabel('Volume (m$^3$/mol)')
    plt.ylabel('Pressure (Pa)')
    plt.title(f'T = {Temp} K and P = {Press} Pa')
    plt.grid()
    
    plt.figure(dpi = fig_dpi)
    plt.loglog(V, P_log, [vol[0]], [Press], 'o', markersize=10)
    plt.xlabel('Volume (m$^3$/mol)')
    plt.ylabel('Pressure (Pa)')
    plt.title(f'T = {Temp} K and P = {Press} Pa')
    plt.grid()

else:
    P_max, V_max, P_min, V_min = pmax_min(P, V)

    if len(vol) == 1:

        if vol[0] > V_max:  # gas phase

            print('V_L = NaN')
            print(f'V_V = {vol[0]} [m^3/mol]')
            
            print('Z_L = NaN')
            print('Z_V =', Z(Press, vol[0], Temp))

            print('H_departure_L  = NaN')
            print(f'H_departure_V = {H_depart(vol[0], Press, Temp, Tempc, Pressc, Omega)} [J/mol]')

            print('S_departure_L = NaN')
            print(f'S_departure_V = {S_depart(vol[0], Press, Temp, Tempc, Pressc, Omega)} [J/mol/K]')

            print('φ_L = NaN')
            print(f'φ_V = {np.exp(lnphi(vol[0], Press, Temp, Tempc, Pressc, Omega))}')

            print('The stable phase is vapor, indicated by the circle on the plots. The liquid phase does not exist at the given T and P.')
            print('Warning: Part of the curve in the logarithmic scale figure may be truncated to prevent errors from displaying negative values.')
        
        else:  # liquid phase

            print(f'V_L = {vol[0]} [m^3/mol]')
            print('V_V = NaN')
            
            print('Z_L =', Z(Press, vol[0], Temp))
            print('Z_V = NaN')
            
            print(f'H_departure_L = {H_depart(vol[0], Press, Temp, Tempc, Pressc, Omega)} [J/mol]')
            print('H_departure_V  = NaN')

            print(f'S_departure_L = {S_depart(vol[0], Press, Temp, Tempc, Pressc, Omega)} [J/mol/K]')
            print('S_departure_V = NaN')

            print(f'φ_L = {np.exp(lnphi(vol[0], Press, Temp, Tempc, Pressc, Omega))}')
            print('φ_V = NaN')

            print('The stable phase is liquid, indicated by the circle on the plots. The vapor phase does not exist at the given T and P.')
            print('Warning: Part of the curve in the logarithmic scale figure may be truncated to prevent errors from displaying negative values.')
        
        plt.figure(dpi = fig_dpi)            
        plt.semilogx(V, P, [vol[0]], [Press], 'o', markersize=10)
        plt.xlabel('Volume (m$^3$/mol)')
        plt.ylabel('Pressure (Pa)')
        plt.title(f'T = {Temp} K and P = {Press} Pa')
        plt.grid()
        
        plt.figure(dpi = fig_dpi)  
        plt.loglog(V, P_log, [vol[0]], [Press], 'o', markersize=10)
        plt.xlabel('Volume (m$^3$/mol)')
        plt.ylabel('Pressure (Pa)')
        plt.title(f'T = {Temp} K and P = {Press} Pa')
        plt.grid()

    else:
        vol_V = max(vol)
        vol_L = min(vol)

        print(f'V_L = {vol_L} [m^3/mol]')
        print(f'V_V = {vol_V} [m^3/mol]')
        
        print('Z_L =', Z(Press, vol_L, Temp))
        print('Z_V =', Z(Press, vol_V, Temp))

        print(f'H_departure_L = {H_depart(vol_L, Press, Temp, Tempc, Pressc, Omega)} [J/mol]')
        print(f'H_departure_V = {H_depart(vol_V, Press, Temp, Tempc, Pressc, Omega)} [J/mol]')

        print(f'S_departure_L = {S_depart(vol_L, Press, Temp, Tempc, Pressc, Omega)} [J/mol/K]')
        print(f'S_departure_V = {S_depart(vol_V, Press, Temp, Tempc, Pressc, Omega)} [J/mol/K]')

        phi_L = np.exp(lnphi(vol_L, Press, Temp, Tempc, Pressc, Omega))
        phi_V = np.exp(lnphi(vol_V, Press, Temp, Tempc, Pressc, Omega))

        print(f'φ_L = {phi_L}')
        print(f'φ_V = {phi_V}')

        if phi_L < phi_V:
            if abs(np.log(phi_L / phi_V)) > 0.05:
                print('The stable phase is liquid, indicated by the circle on the plots. The triangle indicates the unstable vapor phase.')
                print('Warning: Part of the curve in the logarithmic scale figure may be truncated to prevent errors from displaying negative values.')
                
                plt.figure(dpi = fig_dpi)
                plt.semilogx(V, P, [vol_L], [Press], 'o', [vol_V], [Press], '^', markersize=10)
                plt.xlabel('Volume ($m^3$/mol)')
                plt.ylabel('Pressure (Pa)')
                plt.title(f'T = {Temp} K and P = {Press} Pa')
                plt.grid()
                
                plt.figure(dpi = fig_dpi)
                plt.loglog(V, P_log, [vol_L], [Press], 'o', [vol_V], [Press], '^', markersize=10)
                plt.xlabel('Volume (m$^3$/mol)')
                plt.ylabel('Pressure (Pa)')
                plt.title(f'T = {Temp} K and P = {Press} Pa')
                plt.grid()
                
            else:
                print('The liquid phase is in equilibrium with the vapor phase, indicated by the circles on the plots.')
                print('Warning: Part of the curve in the logarithmic scale figure may be truncated to prevent errors from displaying negative values.')
               
                plt.figure(dpi = fig_dpi)
                plt.semilogx(V, P, [vol_L, vol_V], [Press, Press], 'o', markersize=10)
                plt.xlabel('Volume (m$^3$/mol)')
                plt.ylabel('Pressure (Pa)')
                plt.title(f'T = {Temp} K and P = {Press} Pa')
                plt.grid()
                
                plt.figure(dpi = fig_dpi)
                plt.loglog(V, P_log, [vol_L, vol_V], [Press, Press], 'o', markersize=10)
                plt.xlabel('Volume (m$^3$/mol)')
                plt.ylabel('Pressure (Pa)') 
                plt.title(f'T = {Temp} K and P = {Press} Pa')
                plt.grid()
        else:
            if abs(np.log(phi_L / phi_V)) > 0.05:
                print('The stable phase is vapor, indicated by the circle on the plots. The triangle indicates the unstable liquid phase.')
                print('Warning: Part of the curve in the logarithmic scale figure may be truncated to prevent errors from displaying negative values.')
                
                plt.figure(dpi = fig_dpi)
                plt.semilogx(V, P, [vol_V], [Press], 'o', [vol_L], [Press], '^', markersize=10)
                plt.xlabel('Volume (m$^3$/mol)')
                plt.ylabel('Pressure (Pa)')
                plt.title(f'T = {Temp} K and P = {Press} Pa')
                plt.grid()
                
                plt.figure(dpi = fig_dpi)
                plt.loglog(V, P_log, [vol_V], [Press], 'o', [vol_L], [Press], '^', markersize=10)
                plt.xlabel('Volume (m$^3$/mol)')
                plt.ylabel('Pressure (Pa)')
                plt.title(f'T = {Temp} K and P = {Press} Pa')
                plt.grid()
                
            else:
                print('The vapor phase is in equilibrium with the liquid phase, indicated by the circles on the plots.')
                print('Warning: Part of the curve in the logarithmic scale figure may be truncated to prevent errors from displaying negative values.')
                
                plt.figure(dpi = fig_dpi)
                plt.semilogx(V, P, [vol_L, vol_V], [Press, Press], 'o', markersize=10)
                plt.xlabel('Volume (m$^3$/mol)')
                plt.ylabel('Pressure (Pa)')
                plt.title(f'T = {Temp} K and P = {Press} Pa')
                plt.grid()
                
                plt.figure(dpi = fig_dpi)
                plt.loglog(V, P_log, [vol_L, vol_V], [Press, Press], 'o', markersize=10)
                plt.xlabel('Volume (m$^3$/mol)')
                plt.ylabel('Pressure (Pa)')
                plt.title(f'T = {Temp} K and P = {Press} Pa')
                plt.grid()
    
    plt.show()    