######################################################################################
# This code (Version 1.6) is provided for educational purposes.
# If you use, adapt, or redistribute this code, please cite the source as follows:
# Dun-Yen Kang at National Taiwan University, http://www.mmlab-ntu.tw , September 2024 
######################################################################################

import numpy as np
import matplotlib.pyplot as plt

R = 8.314

def kappa(omega):
    return 0.37464 + 1.5422 * omega - 0.26992 * omega**2

def alpha(t, tc, omega):
    return (1 + kappa(omega) * (1 - np.sqrt(t / tc)))**2

def b(tc, pc):
    return 0.0778 * R * tc / pc

def a(t, tc, pc, omega):
    return 0.45724 * R**2 * tc**2 / pc * alpha(t, tc, omega)

def P_PR_EOS(t, tc, pc, omega, v):
    return R * t / (v - b(tc, pc)) - a(t, tc, pc, omega) / (v * (v + b(tc, pc)) + b(tc, pc) * (v - b(tc, pc)))

def pmax_min(p, v): #find the local max and min for the PV curve 
    pks_max = np.r_[True, p[1:] > p[:-1]] & np.r_[p[:-1] > p[1:], True]
    pks_min = np.r_[True, p[1:] < p[:-1]] & np.r_[p[:-1] < p[1:], True]
    pmax = np.sum(p * pks_max)
    vmax = np.sum(v * pks_max)
    pmin = np.sum(p * pks_min)
    vmin = np.sum(v * pks_min)
    return pmax, vmax, pmin, vmin

def finding_vl_vv(p, t, tc, pc, omega): #solve the PREOS in a form of cubic equation at given T, P
    A = a(t, tc, pc, omega) * p / R**2 / t**2
    B = b(tc, pc) * p / R / t
    al = -(1 - B)
    be = (A - 2 * B - 3 * B**2)
    ga = -(A * B - B**2 - B**3)
    roots = np.roots([1, al, be, ga])
    return roots * R * t / p

def Z(p, v, t):
    return p * v / R / t

def lnphi(v, p, t, tc, pc, omega): #compute natural log phi from PREOS
    Z_value = Z(p, v, t)
    b_value = b(tc, pc)

    if Z_value <= 0 or np.isclose(Z_value, b_value * p / R / t):
        # Handle the case when the argument of the logarithm becomes invalid
        return np.nan  # or any other appropriate value in this case

    term1 = (Z_value - 1)
    term2 = np.log(Z_value - b_value * p / R / t)
    term3 = a(t, tc, pc, omega) / b(tc, pc) / R / t / 2.828 * np.log((Z_value + 2.414 * b_value * p / R / t) / (Z_value - 0.414 * b_value * p / R / t))

    return term1 - term2 - term3

def H_depart(v, p, t, tc, pc, omega): # compute enthalpy departure function
    dadt = (a(t, tc, pc, omega) - a(t - 0.1, tc, pc, omega)) / 0.1
    cap_B = b(tc, pc) * p / R / t
    term1 = R * t * (Z(p, v, t) - 1)
    term2 = (t * dadt - a(t, tc, pc, omega)) / (2 * np.sqrt(2) * b(tc, pc))
    term3 = np.log((Z(p, v, t) + (1 + np.sqrt(2)) * cap_B) / (Z(p, v, t) + (1 - np.sqrt(2)) * cap_B))
    return term1 + term2 * term3

def S_depart(v, p, t, tc, pc, omega): # compute entropy departure function
    dadt = (a(t, tc, pc, omega) - a(t - 0.1, tc, pc, omega)) / 0.1
    cap_B = b(tc, pc) * p / R / t
    term1 = R * np.log(Z(p, v, t) - cap_B)
    term2 = dadt / (2 * np.sqrt(2) * b(tc, pc))
    term3 = np.log((Z(p, v, t) + (1 + np.sqrt(2)) * cap_B) / (Z(p, v, t) + (1 - np.sqrt(2)) * cap_B))
    return term1 + term2 * term3

Tempc = float(input('Enter critical temperature in [K] of the target fluid (154.6 for Oxygen): '))
Pressc = float(input('Enter critical pressure in [Pa] of the target fluid (5.046e6 for Oxygen): '))
Omega = float(input('Enter acentric factor, ω, of the target fluid (0.021 for Oxygen): '))
Temp = float(input('Specify the VLE temperature in [K] (a temperture below Tc): '))
# DPI of the figures
fig_dpi = int(input('Specify DPI of the output figures (150 is suggested value): '))

Vstart = b(Tempc, Pressc) * 1.05
V = np.logspace(np.log10(Vstart), -1, 500)
P = np.zeros(500)
for i in range(500):
    P[i] = P_PR_EOS(Temp, Tempc, Pressc, Omega, V[i])

P_log = np.zeros(500)
for i in range(500):
    if P[i] < 1E1:
        P_log[i] = 1E1
    else:
        P_log[i] = P[i]

if Temp >= Tempc:
    print('T >= Tc, so VLE does not exist.')
    print('Only supercritical fluid exists at the given temperature.')

else:
    print('VLE exists. See below.')

    P_max, V_max, P_min, V_min = pmax_min(P, V)

    p_eq_guess = np.logspace(2, np.log10(P_max), 5000)
    old_guess = 100

    for i in range(len(p_eq_guess)):
        V_v_guess = np.max(finding_vl_vv(p_eq_guess[i], Temp, Tempc, Pressc, Omega))
        V_l_guess = np.min(finding_vl_vv(p_eq_guess[i], Temp, Tempc, Pressc, Omega))

        ln_phi_v_guess = lnphi(V_v_guess, p_eq_guess[i], Temp, Tempc, Pressc, Omega)
        ln_phi_l_guess = lnphi(V_l_guess, p_eq_guess[i], Temp, Tempc, Pressc, Omega)

        if abs(ln_phi_v_guess - ln_phi_l_guess) <= old_guess:
            old_guess = abs(ln_phi_v_guess - ln_phi_l_guess)
            P_eq = p_eq_guess[i]
            V_eq_v = V_v_guess
            V_eq_l = V_l_guess
            Z_eq_v = Z(P_eq, V_eq_v, Temp)
            Z_eq_l = Z(P_eq, V_eq_l, Temp)
            phi_v = np.exp(ln_phi_v_guess)
            phi_l = np.exp(ln_phi_l_guess)

    print('P_eq =', P_eq, '[Pa]')
    
    print('V_L =', V_eq_l, '[m^3/mol]')
    print('V_V =', V_eq_v, '[m^3/mol]')
    
    print('Z_L =', Z_eq_l)
    print('Z_V =', Z_eq_v)
    
    print(f'H_departure_L = {H_depart(V_eq_l, P_eq, Temp, Tempc, Pressc, Omega)} [J/mol]')
    print(f'H_departure_V = {H_depart(V_eq_v, P_eq, Temp, Tempc, Pressc, Omega)} [J/mol]')
    
    print(f'S_departure_L = {S_depart(V_eq_l, P_eq, Temp, Tempc, Pressc, Omega)} [J/mol/K]')
    print(f'S_departure_V = {S_depart(V_eq_v, P_eq, Temp, Tempc, Pressc, Omega)} [J/mol/K]')
    
    print('φ_L =', phi_l)
    print('φ_V =', phi_v)
    

    if abs(np.log(phi_l / phi_v)) < 0.01:
        print('This VLE calculation is reliable. The fugacity coefficients of the two phases are close enough.')
    else:
        print('This VLE calculation is NOT reliable! The fugacity coefficients differ too much.')

    print('Warning: Part of the curve in the logarithmic scale figure may be truncated to prevent errors from displaying negative values.')
    
    # Plotting the results with larger dots (e.g., s=50 for larger dots)
    plt.figure(dpi = fig_dpi)
    plt.semilogx(V, P, zorder = 1)
    plt.scatter([V_eq_v, V_eq_l], [P_eq, P_eq], marker='o', s=80, zorder = 2)
    plt.xlabel('Volume (m^3/mol)')
    plt.ylabel('Pressure (Pa)')
    plt.title(f'Isotherm at {Temp} K')
    plt.grid()
    label = 'test'
    # Scatter plot after the curve to ensure dots sit on top

    plt.figure(dpi = fig_dpi)
    plt.loglog(V, P_log, zorder = 1)
    plt.scatter([V_eq_v, V_eq_l], [P_eq, P_eq], marker='o', s=80, zorder = 2)
    plt.xlabel('Volume (m$^3$/mol)')
    plt.ylabel('Pressure (Pa)')
    plt.title(f'Isotherm at {Temp} K')
    plt.grid()
    # Scatter plot after the curve to ensure dots sit on top

    plt.show()